% Figures_9_10.m
% 
% baseline results figures
% 
% "The Past and Future of U.S. Structural Change" 
% Andrew Foerster, Andreas Hornstein, Pierre-Daniel Sarte, Mark Watson
% September 2025
% % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % 

% -- Clear Workspace -- %
restoredefaultpath;
clear;
clc;
close all;

% -- File Directories -- % 
datadir = 'Data\DataReplication\DataFinal\'; 
outdir  = 'Tables\';
figdir  = 'Figures\';
% figdir  = 'Figures\Robustness\LL_'; warning('Low Elasticity Results'); pause
% figdir  = 'Figures\Robustness\HH_'; warning('High Elasticity Results'); pause
matdir  = 'Output\';

% -- Add paths -- %
addpath('Functions\');
addpath('Utilities\')
addpath('Data\')

% -- Options -- %
options = optimset('Display','off');
T = 72;
nS = 5;
disp_code = 0;  % not displaying convergence every iteration
q = 7;


% -- Create Matrices -- %
va_share_model  = NaN(T,nS);
va_share_data   = NaN(T,nS);
theta_model     = NaN(T,nS);
Phi_model       = NaN(nS,nS,T);
gamma_model     = NaN(T,nS);
Omega_model     = NaN(nS,nS,T);
psix_model      = NaN(T,nS);
psi_model       = NaN(T,nS);
py_model        = NaN(T,nS);
pv_model        = NaN(T,nS);
v_model         = NaN(T,nS);
c_model         = NaN(T,nS);
e_model         = NaN(T,1);
en_model        = NaN(T,1);
rho_model       = NaN(T,1);
lambdac_model   = NaN(T,1);
varphi_model    = NaN(T,nS);
cn_model        = NaN(T,1);
ck_model        = NaN(T,1);
w_model         = NaN(T,nS);

% -- Load Data -- %
load_data;
py = y ./ yq;
pv = v ./ vq;
sq = 7;                              % q = 7, corresponds approximately to a 20 year trend
I_trend = 1;
trend_py        = LF_Proj(py,q,I_trend);
load([matdir 'driving_processes']);
load([matdir 'psi_process'])
c   = (squeeze(ee(:,1,:)))';    % c is nominal expenditures on consumption, ee contains final demand of PCE, Private Fixed Inv, Gov Cons, Gov Fixed Inv, and Inv + NX produced by each sector 
E   = (sum(c(:,:),2));          % total consumption expenditures
eta_data = (pv.*vq)./E;

% -- Construct TFP Level -- %
model = model_setup(1);         % calling model parameters
ratio_en_pyb = model.ratio_en_pyb;
s_c = model.s_c;
e_normalized    = (ratio_en_pyb * 1)/(1-s_c);      % Normalized Total Consumption Expenditures
z = find_tfp_scale(py(1,:)'./py(1,4),trend_labor(1,:)',eta_data(1,:)',e_normalized,model);
tfp = zeros(T,nS);
tfp(1,:) = z(:,1);
for i = 2:T
    tfp(i,:) = tfp(i-1,:).*exp(d_tfp(i,:));
end
trend_tfp   = LF_Proj(tfp,q,1);

save([matdir 'tfp_process'],'tfp','trend_tfp');


% inital guess
py_guess = trend_py(1,:)'/trend_py(1,4);

tic
for tt = 1:T
    disp(['t = ' num2str(tt)]);
    disp(' ')
    disp(' ')
    
    % -- Setting Up Model Parameters and Driving Processes -- %
    model = model_setup(tt);


    % -- Solve for the Equilibrium -- %
    [equilibrium,retcode] = fcn_steps_solver(trend_labor(tt,:)',disp_code,py_guess,trend_tfp(tt,:)',psi_nxg_trend(tt,:)',model);
    
    % -- Outputs from procedure -- %
    py      = equilibrium.py;
    pv      = equilibrium.pv;
    px      = equilibrium.px;
    pm      = equilibrium.pm;
    y       = equilibrium.y;
    v       = equilibrium.v;
    xij     = equilibrium.xij;
    x       = equilibrium.x;
    mij     = equilibrium.mij;
    m       = equilibrium.m;
    c       = equilibrium.c;
    e       = equilibrium.e;
    en      = equilibrium.en;
    psi     = equilibrium.psi;
    z       = equilibrium.z;
    varphi  = equilibrium.varphi;
    lambdac = equilibrium.lambdac;
    rho     = equilibrium.rho;
    cn      = equilibrium.cn;
    ck      = equilibrium.ck;
    w       = equilibrium.w;

    % -- Model Consistency Checks -- %
    disp('Model Consistency Checks:')
    ind1_inv    = model.ind1_inv;
    ind2_inv    = model.ind2_inv;
    eta_x       = model.eta_x;
    epsilon_x   = model.epsilon_x;
    rho_x       = model.rho_x;
    zeta_x      = model.zeta_x;
    ind1_mat    = model.ind1_mat;
    ind2_mat    = model.ind2_mat;
    eta_m       = model.eta_m;
    epsilon_m   = model.epsilon_m;
    rho_m       = model.rho_m;
    zeta_m      = model.zeta_m;
    gamma_y     = model.gamma_y;
    beta        = model.beta;
    delta       = model.delta;
    alpha       = model.alpha;
    s_c         = model.s_c;
    ind1_c      = model.ind1_c;
    ind2_c      = model.ind2_c;
    zeta_c      = model.zeta_c;
    Theta_c     = model.Theta_c;
    epsilon_c   = model.epsilon_c;
    sigma       = model.sigma;
    gamma_l     = model.gamma_l;

    
    % -- Investment Input Shares -- %
    inv_input_shares_model          = NaN(nS,nS);
    Omega_model_direct              = NaN(nS,nS);
    for j = 1:nS
        for i = 1:nS
            Omega_model_direct(i,j) = py(i)*xij(i,j)/px(j)/x(j);
        end
        for i = ind1_inv
            xkj = sum(eta_x(ind1_inv,j).*xij(ind1_inv,j).^((epsilon_x(j)-1)/epsilon_x(j))).^(epsilon_x(j)/(epsilon_x(j)-1));
            inv_input_shares_model(i,j)  = xkj^((1-epsilon_x(j))/epsilon_x(j))*xij(i,j)^((epsilon_x(j)-1)/epsilon_x(j))*eta_x(i,j)*rho_x(j);
        end
        for i = ind2_inv
            inv_input_shares_model(i,j)  = zeta_x(i,j)*(1-rho_x(j));
        end
    end
    inv_share_check = norm(inv_input_shares_model-Omega_model_direct);    
    disp(['Investment Shares = ' num2str(inv_share_check)])
    if inv_share_check > 1e-6
        warning('Inaccuracy in Investment Shares')
        pause;
    end


    % -- Material Input Shares -- %
    mat_input_shares_model          = NaN(nS,nS);
    Phi_model_direct                = NaN(nS,nS);
    for j = 1:nS
        for i = 1:nS
            Phi_model_direct(i,j) = py(i)*mij(i,j)/pm(j)/m(j);
        end
        for i = ind1_mat
            mkj = sum(eta_m(ind1_mat,j).*mij(ind1_mat,j).^((epsilon_m(j)-1)/epsilon_m(j))).^(epsilon_m(j)/(epsilon_m(j)-1));
            mat_input_shares_model(i,j)  = mkj^((1-epsilon_m(j))/epsilon_m(j))*mij(i,j)^((epsilon_m(j)-1)/epsilon_m(j))*eta_m(i,j)*rho_m(j);
        end
        for i = ind2_mat
            mat_input_shares_model(i,j)  = zeta_m(i,j)*(1-rho_m(j));
        end
    end
    mat_share_check = norm(mat_input_shares_model-Phi_model_direct);    
    disp(['Material Shares = ' num2str(mat_share_check)])
    if mat_share_check > 1e-6
        warning('Inaccuracy in Material Shares')
        pause;
    end
    
    
    % -- Value Added Shares -- %
    va_shares_model         = NaN(nS,1);
    gamma_model_direct  = NaN(nS,1);
    for j = 1:nS
        gamma_model_direct(j) = pv(j)*v(j)/py(j)/y(j);
    end
    for j = 1:nS
        va_shares_model(j)  = gamma_y(j);
    end
    va_share_check = norm(va_shares_model-gamma_model_direct);    
    disp(['Value Added Shares in Gross Output = ' num2str(va_share_check)])
    if va_share_check > 1e-6
        warning('Inaccuracy in Value Added Shares in Gross Output')
        pause;
    end

    
    % -- Value Added Shares -- %
    con_shares_model         = NaN(nS,1);
    theta_model_direct  = NaN(nS,1);
    for j = 1:nS
        theta_model_direct(j) = py(j)*c(j)/e;
    end
    for j = ind1_c
        con_shares_model(j) = zeta_c(j)*s_c;
    end
    omegac = zeros(nS,1);
    for j = ind2_c(2:end)
        pyb = py(ind2_c(1));        % 
        En = sum(py(ind2_c).*c(ind2_c));
        fun = @(omega) Theta_c(j)*(py(j)/pyb)^(1-sigma)*(En/pyb)^((1-sigma)*(epsilon_c(j)-1))*(1-omega)^(epsilon_c(j)) - omega;
        omegac(j) = fsolve(fun,.2,options);
        con_shares_model(j) = omegac(j)*(1-s_c);
    end
    
    for j = ind2_c(1)
        omegac(j) = 1 - sum(omegac);
        con_shares_model(j) = omegac(j)*(1-s_c);
    end
    con_share_check = norm(con_shares_model-theta_model_direct);    
    disp(['Consumption Shares = ' num2str(con_share_check)])
    if con_share_check > 1e-6
        warning('Inaccuracy in Consumption Shares')
        pause;
    end

    % Government and Net Exports
    psix_model_direct   = px.*x./(pv.*v);
    
    % -- Data vs Model Comparisons -- %
    disp(' ')
    disp(' ')
    disp('Data vs Model Comparison:')
    % Prices
    py_norm = py./py(4);
    disp('Trend Prices')
    disp('      Data    Model   Model with NDG as numeraire')    
    disp([trend_py(tt,:)'/trend_py(tt,4) py py_norm])

    % Value Added Shares
    % trend_va_shares_data = trend_sv(tt,:)'*100;
    eta_psi = ((eye(nS) - Phi_model_direct*(eye(nS)-diag(gamma_model_direct)))*inv(diag(gamma_model_direct)) - Omega_model_direct*diag(psix_model_direct)   - diag(psi))\theta_model_direct; %#ok<*MINV>
    va_shares_model_psi = (eta_psi/sum(eta_psi))*100;
    % disp('Trend Value Added Shares')
    % disp('      Data    Model')
    % disp([trend_va_shares_data va_shares_model_psi])


    % save model values
    va_share_model(tt,:) = va_shares_model_psi/100;
    % va_share_data(tt,:)  = trend_va_shares_data/100;
    theta_model(tt,:)    = theta_model_direct';
    Phi_model(:,:,tt)    = Phi_model_direct;
    gamma_model(tt,:)    = gamma_model_direct';
    Omega_model(:,:,tt)  = Omega_model_direct;
    psix_model(tt,:)     = psix_model_direct';
    psi_model(tt,:)      = psi';
    py_model(tt,:)       = py';
    pv_model(tt,:)       = pv';
    v_model(tt,:)        = v';
    e_model(tt)          = e;
    c_model(tt,:)        = c;
    en_model(tt)         = en;
    rho_model(tt,:)      = rho;    
    lambdac_model(tt,:)  = lambdac;
    varphi_model(tt,:)   = varphi;
    cn_model(tt,:)        = cn;
    ck_model(tt,:)        = ck;
    w_model(tt,:)         = w;

    py_guess = py;


    disp(' ')
    disp(' ')
    disp(' ')
    disp('-----------------')
end
toc

save('Output/model_output')


%% -- MODEL RESULTS -- %

% load('Data/Data_5_sec.mat','SecName','nS','T','trend_psi','psi','sv_eq_raw',...
    % 'sv_nopsi','theta','omega','phi','colors',...
    % 'pie_chart_names','years')


% -- Building Value Added Shares from Matrix Equation-- %
psi_x_d     = build_diag(psiX);
psi_nx_d    = build_diag(psiNX);
psi_g_d     = build_diag(psiG);
gamma_d     = build_diag(gamma);

eta_eq_raw = zeros(T,nS);
for ii = 1:T
        eta_eq_raw(ii,:) =  (inv((eye(nS) - (phi(:,:,ii)*(eye(nS) - gamma_d(:,:,ii)))) * inv(gamma_d(:,:,ii)) ...
            - omega(:,:,ii)*psi_x_d(:,:,ii) - psi_g_d(:,:,ii) - psi_nx_d(:,:,ii) ) * theta(ii,:).')'; %#ok<*MINV>
end
sv_eq_raw = eta_eq_raw ./ (sum(eta_eq_raw,2));

% Figure 2 with Model Output
gamma_d_model = build_diag(gamma_model);
model_psi_x_d = build_diag(psix_model);

% Adding in capital
eta_model = zeros(nS,T);
for tt = 1:T
    eta_model(:,tt) =  inv((eye(nS) - (Phi_model(:,:,tt)*(eye(nS) - gamma_d_model(:,:,tt)))) * inv(gamma_d_model(:,:,tt)) - Omega_model(:,:,tt)*model_psi_x_d(:,:,tt)) ...
                        * theta_model(tt,:).';
end
eta_model = eta_model.';
sv_model_2 = eta_model ./ (sum(eta_model,2));

% Creating value added shares with psi = 0
eta_nopsi = zeros(T,nS);
for ii = 1:T
    eta_nopsi(ii,:) =  (inv((eye(nS) - (phi(:,:,ii)*(eye(nS) - gamma_d(:,:,ii)))) * inv(gamma_d(:,:,ii)) - omega(:,:,ii)*psi_x_d(:,:,ii)) ...
                        * theta(ii,:).')';
end
sv_nopsi = eta_nopsi ./ (sum(eta_nopsi,2));



ylims = [0 0.1; 0 0.2; 0 0.2; 0 0.4; 0.4 0.8; 0 0.5];

i_loc = [3 1 2 4 5];
calvec = years;
for i = 1:nS
    figure('Units','Inches','Position',[1 1 2 2]);
    hold on;
    plot(calvec,va_share_model(:,i),'- k','LineWidth',3)
    plot(calvec,theta_model(:,i),'-- r','LineWidth',3);
    plot(calvec,sv_model_2(:,i),': b','LineWidth',4);
        
    tmp = LF_Proj(sv_eq_raw(:,i),q,1);
    plot(calvec,tmp,'- k','LineWidth',1);
    tmp = LF_Proj(theta(:,i),q,1);
    plot(calvec,tmp,'-- r','LineWidth',1);
    tmp = LF_Proj(sv_nopsi(:,i),q,1);
    plot(calvec,tmp,': b','LineWidth',1.5);

    hold off;
    ylim([0.0 0.20]);
    xlim([1940 2020]);
    if i_loc(i) == 5
        ylim([0.45 0.85]);
    elseif i_loc(i) == 4
        ylim([0.0 0.40]);
    end
    
    % if i_loc(i) == 1
    %    legend('Value Added Shares','Consumption Shares','Value Added Shares (Excl. NX & G)','Location','NorthWest','FontSize',ax_fs);
    % end
    grid on;
    box on
    print('-depsc',[figdir 'Figure_9_' num2str(i_loc(i))])
end

% Add in Sum of Const, Durables and IPP
figure('Units','Inches','Position',[1 1 2 2]);
hold on;
    plot(calvec,sum(va_share_model(:,1:3),2),'- k','LineWidth',3)
    plot(calvec,sum(theta_model(:,1:3),2),'-- r','LineWidth',3);
    plot(calvec,sum(sv_model_2(:,1:3),2),': b','LineWidth',4);
    tmp = LF_Proj(sum(sv_eq_raw(:,1:3),2),q,1);
    plot(calvec,tmp,'- k','LineWidth',1);
    tmp = LF_Proj(sum(theta(:,1:3),2),q,1);
    plot(calvec,tmp,'-- r','LineWidth',1);
    tmp = LF_Proj(sum(sv_nopsi(:,1:3),2),q,1);
    plot(calvec,tmp,': b','LineWidth',1.5);
hold off;
ylim([0.0 0.40]);
xlim([1940 2020]);
% title('Durables+IPP+Const.','FontSize',ax_fs);
% ax = gca;
% ax.FontSize = ax_fs;
grid on;
box on;
print('-depsc',[figdir 'Figure_9_6'])
% fname = ['Figure_9'];
% figure_to_png(figdir,fname);
% close(fig); 





% -- Figure 10 -- %
% Indices for sectors
i_dur = 2;
i_ndur = 4;
i_serv = 5;
i_const = 1;
i_ipp = 3;
for i = 1:6
    figure('Units','Inches','Position',[1 1 2 2]);
    if i == 1
        TMP = Omega_model;
        TMP1 = omega;
        jj = i_dur;
        tit_str = 'Durable Goods Investment';
        ylim1 = [0 0.8];
    elseif i == 2
        TMP = Omega_model;
        TMP1 = omega;
        jj = i_ndur;
        tit_str = 'Nondurable Goods Investment';
        ylim1 = [0 0.6];
    elseif i == 3
        TMP = Omega_model;
        TMP1 = omega;
        jj = i_serv;
        tit_str = 'Sevices Investment';
        ylim1 = [0 0.4];
    elseif i == 4
        TMP = Phi_model;
        TMP1 = phi;
        jj = i_dur;
        tit_str = 'Durable Goods Materials';
        ylim1 = [0 0.8];
    elseif i == 5
        TMP = Phi_model;
        TMP1 = phi;
        jj = i_ndur;
        tit_str = 'Nondurable Goods Materials';
        ylim1 = [0 0.1];
    elseif i == 6
        TMP = Phi_model;
        TMP1 = phi;
        jj = i_serv;
        tit_str = 'Services Materials';
        ylim1 = [0 0.2];
    end

    x_dur = squeeze(TMP(i_dur,jj,:));
    x_ipp = squeeze(TMP(i_ipp,jj,:));
    tmp1 = squeeze(TMP1(i_dur,jj,:));
    y_dur = LF_Proj(tmp1,q,1);
    tmp1 = squeeze(TMP1(i_ipp,jj,:));
    y_ipp = LF_Proj(tmp1,q,1);
    plot(calvec,x_dur,'- b','LineWidth',3);
    hold on;
        plot(calvec,x_ipp,'-- r','LineWidth',3);
        plot(calvec,y_dur,'- b','LineWidth',1);
        plot(calvec,y_ipp,'-- r','LineWidth',1);
    hold off;

    ylim(ylim1);
    grid on;
    box on

    print('-depsc',[figdir 'Figure_10_' num2str(i)])
end





%% Plot of rho
fig = figure('Units','Inches','Position',[1 1 6 3]);
hold on
plot(years,rho_model,'- b','LineWidth',2)
hold off
box on
grid on
xlim([1940 2020])
ylim([0.05 0.1])
print(fig, [figdir, 'Appendix_Figure_2'],'-depsc')
close(fig)